<?php
/* --------------------------------------------------------------
   ParcelServiceProvider.inc.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService\Model;

use Gambio\Admin\ParcelService\Interfaces\ParcelService as ParcelServiceInterface;
use Gambio\Core\Language\LanguageId;

/**
 * Class ParcelService
 *
 * @package Gambio\Admin\ParcelService\Model
 */
class ParcelService implements ParcelServiceInterface
{
    /**
     * @var ParcelServiceId
     */
    private $id;
    
    /**
     * @var string
     */
    private $name;
    
    /**
     * @var bool
     */
    private $isDefault;
    
    /**
     * @var ParcelServiceDescriptions
     */
    private $descriptions;
    
    
    /**
     * ParcelService constructor.
     *
     * @param ParcelServiceId           $id
     * @param string                    $name
     * @param bool                      $isDefault
     * @param ParcelServiceDescriptions $descriptions
     */
    private function __construct(
        ParcelServiceId $id,
        string $name,
        bool $isDefault,
        ParcelServiceDescriptions $descriptions
    ) {
        $this->id           = $id;
        $this->name         = $name;
        $this->isDefault    = $isDefault;
        $this->descriptions = $descriptions;
    }
    
    
    /**
     * @param ParcelServiceId           $id
     * @param string                    $name
     * @param bool                      $isDefault
     * @param ParcelServiceDescriptions $descriptions
     *
     * @return ParcelService
     */
    public static function create(
        ParcelServiceId $id,
        string $name,
        bool $isDefault,
        ParcelServiceDescriptions $descriptions
    ): ParcelService {
        return new self($id, $name, $isDefault, $descriptions);
    }
    
    
    /**
     * @inheritDoc
     */
    public function id(): ?int
    {
        return $this->id->value();
    }
    
    
    /**
     * @inheritDoc
     */
    public function name(): string
    {
        return $this->name;
    }
    
    
    /**
     * @inheritDoc
     */
    public function isDefault(): bool
    {
        return $this->isDefault;
    }
    
    
    /**
     * @inheritDoc
     */
    public function url(LanguageId $languageId): string
    {
        return $this->descriptions->url($languageId);
    }
    
    
    /**
     * @inheritDoc
     */
    public function comment(LanguageId $languageId): string
    {
        return $this->descriptions->comment($languageId);
    }
    
    
    /**
     * @inheritDoc
     */
    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
    
    
    /**
     * Array serialization.
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'id'           => $this->id->value(),
            'name'         => $this->name,
            'isDefault'    => $this->isDefault,
            'descriptions' => $this->descriptions->toArray(),
        ];
    }
    
    
    /**
     * Create a new parcel service instance from the provided array.
     *
     * @param array $data
     *
     * @return ParcelService
     */
    public static function createFromArray(array $data): ParcelService
    {
        return self::create(ParcelServiceId::create($data['id'] ?? null),
                            $data['name'],
                            $data['isDefault'],
                            ParcelServiceDescriptions::createFromArray($data['descriptions']));
    }
}